/** @file   WarehouseFactory.h
 * @brief   Declaration of WarehouseFactory class.
 * @version $Revision: 1.4 $
 * @date    $Date: 2006/02/21 20:46:51 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_WAREHOUSEFACTORY_H
#define H_ENG2D_WAREHOUSEFACTORY_H

#include "eng2d_dll.h"
#include <string>
#include <vector>
#include "Size2D.h"
#include "datatypes.h"
#include "eng2d_Exception.h"

namespace eng2d {

// Forward declarations
class DataWarehouse;
class BaseContainer;
class FileList;


/** @class  MWarehouseCreationObserver
 * @brief   An interface for object that observes the warehouse creation
 *          process.
 * @author  Tomi Lamminsaari
 */
class MWarehouseCreationObserver
{
public:
  /** A callback function that gets called each time the factory method
   * has loaded one file.
   * @param   aCurrentStep      The current step. Number of files read
   *                            so far.
   * @param   aTotalSteps       Total number of files being read.
   * @param   aCurrentFilename  Name of the file that was just read.
   */
  virtual void NotifyWarehouseCreation( int aCurrentStep, int aTotalSteps,
                                        const std::string& aCurrentFilename ) = 0;
  
  /** Destructor.
   */
  virtual ~MWarehouseCreationObserver() { }
};


/** @class  WarehouseFactory
 * @brief   A factory class that can be used to initialize DataWarehouse.
 * @author  Tomi Lamminsaari
 *
 * This class contains only static methods so there is no need to create any
 * instances of it.
 */
class DLLIMPORT WarehouseFactory
{
public:

    ///
    /// Constants, datatypes and static methods
    /// =======================================

    /** Construcs a warehouse.
     * @param   aWarehouse      Pointer to warehouse being built.
     * @param   aFiles          The filelist containing the contents of the
     *                          warehouse.
     * @param   aObserver       The object that observes the creation
     *                          progression. 0 by default wich means to no
     *                          observer will be set.
     * @exception xRuntimError  Thrown if creation fails.
     */
    static void CreateWarehouse( DataWarehouse* aWarehouse,
                                 const FileList& aFiles,
                                 MWarehouseCreationObserver* aObserver = 0
                               ) throw(xRuntimeError);

protected:
    ///
    /// Protected interface
    /// ===================
    
    /** Possible input formats of the GraphicsItem data.
     */
    enum ReadInMode {
      /** A single bitmap. */
      ESingleBitmap,
      /** Bitmaps are taken from a grid. */
      EGrid,
      /** Bitmaps are taken from an allegro datafile. */
      EDatafile
    };
    
    /** The general information of the GraphicsContainer is stored as
     * an instance of this.
     */
    struct GraphicContainerHeader {
      /** The format how the bitmaps are being stored. */
      GfxContentType  iContentType;
      /** Uid of the GraphicsItem */
      int             iUid;
      /** Source file. */
      std::string     iFilename;
      /** Grid size. */
      Size2D          iGridSize;
      /** The input mode. */
      ReadInMode      iInputMode;
      /** Number of rotations */
      int             iRotations;
      /** Constructor that initializes the members with valid default values.
       */
      GraphicContainerHeader();
    };
    
    struct GraphicContainerGrid {
      int iColumn;
      int iRow;
    };
      
    
    /** Adds new container to a DataWarehouse.
     * @param   aWarehouse      Pointer to warehouse.
     * @param   aFilename       Name of the file that describes the container
     *                          being added to warehouse.
     * @return  KErrNone if successful.
     */
    static int AppendWarehouse( DataWarehouse* aWarehouse,
                                const std::string& aFilename );

    /**
     */
    static BaseContainer* ReadAnimContainer( int& aUid, std::istream& aIn );
    
    /**
     */
    static BaseContainer* ReadGraphicContainer( int& aUid, std::istream& aIn );
    
    /** Reads the header part of the graphic combine.
     * @param   aIn             The input stream.
     * @param   aHeader         An structure where the header data is read.
     * @return  An errorcode or KErrNone if no errors were encountered.
     */
    static int ReadGraphicContainerHeader( std::istream& aIn,
                                           GraphicContainerHeader& aHeader );
                                         
    /** Reads the grid information of graphic combine.
     * @param   aIn             The input stream.
     * @param   aGridParts      Reference to vector where the grid information
     *                          is put.
     * @return  An errorcode. KErrNone if successful.
     */
    static int ReadGraphicContainerGrid( std::istream& aIn,
                                         std::vector<GraphicContainerGrid>& aGridParts );

protected:
    ///
    /// Data members
    /// ============

};

};
#endif
